const router = require("express").Router();
const UsersService = require("../services/UsersService.js");
const multipart = require("connect-multiparty");
const Verify = require("../services/CustomerVerifyToken.js");

var multipartMiddleware = multipart();
let usersService = new UsersService();

/**
 * @swagger
 * /api/customerapp/vendorDetails/{id}:
 *   get:
 *     summary: Get Vendor By Id
 *     tags: [Customer APP APIs]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the vendor
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendor not found
 *       '500':
 *         description: Internal server error
 */
router.get("/vendorDetails/:id", Verify, async function (req, res) {
  try {
    await usersService.getVendorDetails(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/customerapp/vendorReviews/{id}:
 *   get:
 *     summary: Get Vendor Reviews By vendorId
 *     tags: [Customer APP APIs]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the vendor
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *         description: Page number for pagination
 *       - in: query
 *         name: pageSize
 *         schema:
 *           type: integer
 *         description: Number of items per page
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendor not found
 *       '500':
 *         description: Internal server error
 */
router.get("/vendorReviews/:id", Verify, async function (req, res) {
  try {
    const page = parseInt(req.query.page) || 1;
    const pageSize = parseInt(req.query.pageSize) || 10;
    await usersService.getVendorReviews(req, res, page, pageSize);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/customerapp/vendorAlbums/{id}:
 *   get:
 *     summary: Get Vendor Albums By vendorId
 *     tags: [Customer APP APIs]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the vendor
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *         description: Page number for pagination
 *       - in: query
 *         name: pageSize
 *         schema:
 *           type: integer
 *         description: Number of items per page
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendor not found
 *       '500':
 *         description: Internal server error
 */
router.get("/vendorAlbums/:id", Verify, async function (req, res) {
  try {
    const page = parseInt(req.query.page) || 1;
    const pageSize = parseInt(req.query.pageSize) || 10;
    await usersService.getVendorAlbums(req, res, page, pageSize);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/customerapp/vendorCategories:
 *   get:
 *     summary: Get Vendor Categories By Id
 *     tags: [Customer APP APIs]
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendor Categories not found
 *       '500':
 *         description: Internal server error
 */
router.get("/vendorCategories", Verify, async function (req, res) {
  try {
    await usersService.getVendorCategories(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/customerapp/homescreen/{id}:
 *   get:
 *     summary: Get Homescreen vendors By locationId
 *     tags: [Customer APP APIs]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the location
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendors not found
 *       '500':
 *         description: Internal server error
 */
router.get("/homescreen/:id", Verify, async function (req, res) {
  try {
    await usersService.getHomepageVendors(req, res);
  } catch (e) {
    console.log(e);
  }
});


/**
 * @swagger
 * /api/customerapp/vendors:
 *   get:
 *     summary: Get vendors By locationId && categoryId
 *     tags: [Customer APP APIs]
 *     parameters:
 *       - in: query
 *         name: loc_id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the location
 *       - in: query
 *         name: cat_id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the Category
 *       - in: query
 *         name: page
 *         required: false
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Page number for pagination
 *       - in: query
 *         name: limit
 *         required: false
 *         schema:
 *           type: integer
 *           default: 10
 *         description: Number of vendors per page
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendors not found
 *       '500':
 *         description: Internal server error
 */
router.get("/vendors", Verify, async function (req, res) {
  try {
    await usersService.getCategoryVendors(req, res);
  } catch (e) {
    console.log(e);
  }
});
 


/**
 * @swagger
 * /api/customerapp/searchVendor:
 *   get:
 *     summary: Search vendor by keyword
 *     tags: [Customer APP APIs]
 *     parameters:
 *       - in: query
 *         name: catId
 *         required: false
 *         schema:
 *           type: string
 *         description: Vendor Category ID
 *       - in: query
 *         name: locationId
 *         required: true
 *         schema:
 *           type: string
 *         description: Location ID
 *       - in: query
 *         name: keyword
 *         required: true
 *         schema:
 *           type: string
 *         description: Keyword to search
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response
 *       '400':
 *         description: Bad request
 *       '404':
 *         description: Vendor not found
 *       '500':
 *         description: Internal server error
 */
router.get("/searchVendor", Verify, async function (req, res) {
  try {
    await usersService.searchVendors(req, res);
  } catch (e) {
    console.log(e);
  }
});

/**
 * @swagger
 * /api/customerapp/addReview:
 *   post:
 *     summary: Add review
 *     tags: [Customer APP APIs]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               userId:
 *                 type: integer
 *               vendorId:
 *                 type: integer
 *               feedback:
 *                 type: string
 *               rating:
 *                 type: integer
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Review added successfully
 *       '400':
 *         description: Bad request
 *       '500':
 *         description: Internal server error
 */
router.post("/addReview", Verify, async function (req, res) {
  try {
    await usersService.addReview(req, res);
  } catch (e) {
    console.log(e);
  }
});

module.exports = router;
