const router = require('express').Router();
const Response = require('../../models/Response.js');
const ViewsService = require("../../services/vendorServices/ViewsService.js");
const Verify = require('../../services/VendorVerifyToken.js');
const CustomerVerify = require('../../services/CustomerVerifyToken.js');

const viewsService = new ViewsService();
/**
 * @swagger
 * tags:
 *   name: Views
 *   description: API for managing Views
 */

/**
 * @swagger
 * /api/views/getViews:
 *   get:
 *     summary: Get user counts by vendor ID and month
 *     tags: [Views]
 *     description: Retrieves user counts by vendor ID for a specific month.
 *     parameters:
 *       - in: query
 *         name: vendor_id
 *         description: ID of the vendor
 *         required: true
 *         schema:
 *           type: string
 *       - in: query
 *         name: month
 *         description: Month in YYYY-MM format
 *         required: true
 *         schema:
 *           type: string
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Successful response with user counts
 *       '400':
 *         description: Bad request if parameters are missing or invalid
 *       '500':
 *         description: Internal server error
 */

router.get('/getViews', Verify, async function (req, res) {
    await viewsService.getUserCountByVendor(req, res)
});

/**
 * @swagger
 * /api/views/addViews:
 *   post:
 *     summary: Add views for a vendor
 *     tags: [Views] 
 *     description: Add multiple views associated with a specific vendor.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               vendorId:
 *                 type: integer
 *                 description: The ID of the vendor.
 *               viewsData:
 *                 type: array
 *                 description: An array of view objects.
 *                 items:
 *                   type: object
 *                   properties:
 *                     user_id:
 *                       type: integer
 *                       description: The ID of the user associated with the view.
 *     security:
 *       - apiKeyAuth: []  # Apply the apiKeyAuth security scheme to this endpoint
 *     responses:
 *       '200':
 *         description: Views added successfully.
 *       '400':
 *         description: Bad request. Missing vendor ID or views data.
 *       '403':
 *         description: Forbidden. Error submitting views.
 *       '500':
 *         description: Internal server error.
 */
router.post('/addViews', CustomerVerify, async function (req, res) {
    await viewsService.addViews(req, res)
});

module.exports = router;
