const { Sequelize, Op, fn, col, literal } = require("sequelize");
const { vendors, addresses, vendorMedia, vendorReviews, questions, albums, locations, vendorsAnswers, vendorCatBridge, vendorsLockedDates, vendorSubscription, sequelize, vendorPackages, users } = require("../../models/dbCon");
const { uploadFiles, uploadFile } = require("../FileUploadService");
const { genResponse, isEmpty, getLimitForPagination, genPagingResponse, ensureArray } = require("../../commons/commons");

const { HTTP_STATUS_CODES, RESPONSE_MSG } = require("../../commons/constants");
const e = require("express");
class VendorService {
  async getVendors(req, res) {
    try {
      const { catId, page, pageSize, locationId } = req.query || {};
      const { start, end } = getLimitForPagination(page, pageSize);
      let options = {};

      if (page && page > 1 && pageSize) {
        options = {
          limit: end,
          offset: start,
        };
      }
      const condition = {
        [Op.and]: [], // Start with an empty array for conditions
      };
      if (!isEmpty(catId)) {
        condition[Op.and].push({ category_id: catId });
      }

      options.include = [
        {
          model: vendorMedia,
          attributes: ["path"],
          where: { is_banner: 1 },
          required: false,
        },
        {
          model: vendorReviews,
          attributes: [[sequelize.fn("AVG", sequelize.col("rating")), "avgRating"]],
          duplicating: false,
          required: false,
          where: { status: 1 },
        },
      ];

      if (!isEmpty(locationId)) {
        options.include.push({
          model: addresses,
          where: { location_id: locationId },
          attributes: ["full_address", "lat", "long"],
          required: true,
        });
      } else {
        options.include.push({
          model: addresses,
          attributes: ["full_address", "lat", "long"],
        });
      }

      options.where = condition;
      options.attributes = ["id", "name", "slug", "summary"];

      // Adding GROUP BY clause to include necessary columns
      options.group = ["vendors.id", "vendorMedia.id", "vendorReviews.vendor_id", "addresses.id"];

      await sequelize.query("SET sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''));");

      let totalRecords = await vendors.count({ where: condition });
      if (totalRecords == 0) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NORECORD, []);
      }
      let result = await vendors.findAll(options);
      if (page && pageSize) {
        delete options.offset;
        delete options.limit;
        delete options.attributes;
        const totalCount = await vendors.count({ where: condition });
        return genPagingResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result, page, totalCount, pageSize);
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      console.log(e);
    }
  }

  async getVendorDetailByVendorId(req, res) {
    try {
      const vendorId = req.params.id;
      let options = {};

      const condition = {
        [Op.or]: [{ id: vendorId }, { slug: vendorId }],
        [Op.and]: [{ active: true }],
      };

      options.attributes = ["name", "slug", "phone", "business_name", "business_phone", "facebook_page_link", "instagram_page_link", "subType", "website_link", "id", "createdAt"];

      options.where = condition;

      options.include = [
        {
          model: vendorCatBridge,
          attributes: ["vendor_category_id"],
          where: { is_parent: true },
          required: false,
        },
        {
          model: vendorSubscription,
          attributes: ["plan", "sub_plan_id", [literal("DATE(start_date)"), "start_date"], [literal("DATE(expire_date)"), "expire_date"]],
          required: false,
          where: { is_paid: 1 },
        },
        {
          model: addresses,
          attributes: ["full_address", "lat", "long"],
          include: [
            {
              model: locations,
              attributes: ["name"],
              required: false,
            },
          ],
          required: false,
        },
        {
          model: vendorMedia,
          attributes: ["id", "path", "sequence", "is_banner", "image_type", "createdAt"],
          where: {
            [Op.or]: [{ image_type: "cover" }, { image_type: "profile" }],
          },
          required: false,
          order: [["id", "DESC"]],
        },

        {
          model: vendorPackages,
          attributes: ["id", "name", "price", "booking_price", "details"],
          required: false,
        },
      ];

      // await sequelize.query("SET sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''));");
      // let totalRecords = await vendors.count(options)
      // if (totalRecords == 0) {
      //   return genResponse(
      //     res,
      //     HTTP_STATUS_CODES.NOT_FOUND,
      //     RESPONSE_MSG.NORECORD,
      //     []
      //   );
      // }
      let vendorDetails = await vendors.findAll(options);
      // Now fetch vendor reviews separately
      let reviewOptions = {
        attributes: [
          [sequelize.fn("AVG", sequelize.col("rating")), "avgRating"],
          [sequelize.fn("COUNT", sequelize.col("*")), "reviewCount"], // Counting the number of reviews
        ],
        duplicating: false,
        required: false,
        where: { status: 1, vendor_id: vendorId }, // Combining where conditions
      };
      let vendorRev = await vendorReviews.findOne(reviewOptions);
      // // Merge the reviews into vendor details
      if (vendorDetails && vendorRev) {
        vendorDetails.forEach((detail) => {
          detail.dataValues.avgRating = vendorRev.dataValues.avgRating;
          detail.dataValues.reviewCount = vendorRev.dataValues.reviewCount;
        });
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, vendorDetails);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  }

  async getQuestionByvendorCateId(req, res) {
    try {
      const CateId = req.params.CateId;
      const vendorId = req.query.vendorId; // Assuming vendorId is passed in the request parameters
      // Check if any required field is missing
      if (!CateId || !vendorId) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      }
      const result = await questions.findAll({
        include: [
          {
            model: vendorsAnswers,
            attributes: [["answers", "answer"]],
            where: { vendor_id: vendorId },
            required: false,
          },
        ],
        where: {
          vendor_category_id: CateId, // Assuming vendor_category_id is the column representing the vendor ID in your questions table
        },
      });

      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not found");
      }

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  }

  async submitAnswers(req, res) {
    try {
      const { Answers, vendorId } = req.body;

      // Check if any required field is missing
      if (!Answers || Answers.length === 0) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      }

      const results = [];
      for (const answer of Answers) {
        // Check if an answer already exists for the vendor and question combination
        const existingAnswer = await vendorsAnswers.findOne({
          where: {
            vendor_id: vendorId,
            question_id: answer.question_id,
          },
        });

        if (existingAnswer) {
          // If the answer already exists, update it
          await existingAnswer.update({
            answers: answer.answers,
            status: true, // Assuming status is set to true for all updated answers
          });
          results.push(existingAnswer);
        } else {
          // If the answer doesn't exist, create a new one
          const newAnswer = await vendorsAnswers.create({
            answers: answer.answers,
            vendor_id: vendorId,
            question_id: answer.question_id,
            status: true, // Assuming status is set to true for all new answers
          });
          results.push(newAnswer);
        }
      }

      if (results.length > 0) {
        return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, results);
      } else {
        return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, "INTERNAL SERVER ERROR", results);
      }
    } catch (error) {
      // Handle any internal server errors
      console.error("Error submitting answers:", error);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, "Error submitting answers");
    }
  }

  async getsubPlanByvendorId(req, res) {
    try {
      const vendorId = req.params.Id; // Assuming vendorId is passed in the request parameters
      let result = await vendorSubscription.findAll({
        where: {
          vendor_id: vendorId, // Assuming vendor_id is the column representing the vendor ID in your questions table
        },
      });
      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not found");
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  }

  async getlockedDatesByvendorId(req, res) {
    try {
      const vendorId = req.params.Id; // Assuming vendorId is passed in the request parameters
      const currentDate = new Date();
      const next12Months = new Date(currentDate);
      next12Months.setMonth(next12Months.getMonth() + 12);
      // Extracting only the year, month, and date parts of the dates
      const currentDateString = new Date(currentDate.getFullYear(), currentDate.getMonth(), currentDate.getDate());
      const next12MonthsString = new Date(next12Months.getFullYear(), next12Months.getMonth(), next12Months.getDate());

      let whereCondition = {
        vendor_id: vendorId,
        date: {
          [Op.gte]: currentDateString, // Date greater than or equal to current date
          [Op.lte]: next12MonthsString, // Date less than or equal to next 12 months
        },
      };

      const result = await vendorsLockedDates.findAll({
        attributes: ["date", "day", "night", "vendor_id"],
        where: whereCondition,
      });

      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not found");
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      console.error("Error fetching locked dates:", e);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, "Error fetching locked dates");
    }
  }

  async updateLockedDatesByVendorId(req, res) {
    try {
      const { vendorId, date, day, night } = req.body;

      if (!vendorId || !date) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      }
      const whereCondition = {
        vendor_id: vendorId,
        [Op.and]: [sequelize.where(sequelize.fn("DATE", sequelize.col("date")), date), { active: true }],
      };

      const existingRecord = await vendorsLockedDates.findOne({
        where: whereCondition,
      });

      if (!existingRecord) {
        // If no record exists, insert a new one
        const newRecord = {
          vendor_id: vendorId,
          date: date,
          day: day,
          night: night,
          // Add more fields as needed
        };

        await vendorsLockedDates.create(newRecord);
        return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, "New record inserted successfully");
      }

      // If record(s) exist, update them
      const updatedData = {
        day: day,
        night: night,
        // Add more fields to update if needed
      };

      const [numberOfAffectedRows, affectedRows] = await vendorsLockedDates.update(updatedData, {
        where: whereCondition,
      });
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, affectedRows);
    } catch (e) {
      console.error("Error updating/inserting locked dates:", e);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, "Error updating/inserting locked dates");
    }
  }

  async getAlbumByvendorId(req, res) {
    try {
      const vendorId = req.params.Id; // Assuming vendorId is passed in the request parameters

      let result = await albums.findAll({
        attributes: ["id", "name", [sequelize.literal("(SELECT COUNT(*) FROM vendor_media WHERE vendor_media.album_id = albums.id AND active=true)"), "number_of_photo"]],
        include: [
          {
            model: vendorMedia,
            attributes: ["path", "sequence", "is_banner", "image_type"],
            required: false,
            where: {
              image_type: "general",
            },
            limit: 1, // Limiting to one record
          },
        ],
        where: {
          vendor_id: vendorId,
        },
      });

      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not found");
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  }

  async getAlbumDetailByAlbumId(req, res) {
    try {
      const albumId = req.params.Id; // Assuming albumId is passed in the request parameters
      let result = await albums.findOne({
        include: [
          {
            model: vendorMedia,
            attributes: ["id", "path", "sequence", "is_banner", "image_type"],
            required: false,
          },
        ],
        where: {
          id: albumId,
        },
      });

      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not found");
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  }

  async createAlbums(req, res) {
    try {
      const { name, vendor_Id } = req.body;

      // Check if any required field is missing
      if (!name || !vendor_Id) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      }
      // Validate the format or content of name, vendor_id if needed
      // Create a new entry in the albums table
      else {
        const result = await albums.create({
          name: name,
          vendor_id: vendor_Id,
          status: true, // Assuming status is set to true for all new albums
        });

        return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
      }
    } catch (error) {
      // Handle any internal server errors
      console.error("Error submitting albums:", error);
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, "Error submitting albums");
    }
  }

  uploadMedia = async (req, res) => {
    try {
      if (!req.body.vendorId || !req.files || !req.body.imageType) return genResponse(res, HTTP_STATUS_CODES.ERROR, RESPONSE_MSG.RPM);
      const files = ensureArray(req.files.files);
      let sequence = 1;
      for (const file of files) {
        const subvendorId = req.body.subvendorId !== undefined ? req.body.subvendorId : null;
        const albumIdId = req.body.albumId !== undefined ? req.body.albumId : null;

        const uploadMediaName = await uploadFile(file.originalFilename, file.path);
        vendorMedia.create({
          vendor_id: req.body.vendorId,
          album_id: albumIdId,
          subvendor_id: subvendorId,
          image_type: req.body.imageType,
          path: uploadMediaName,
          type: file.type,
          sequence: sequence,
        });
        sequence++;
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, "Files uploaded successfully");
    } catch (e) {
      console.log(e);
    }
  };

  deleteVendorMedia = async (req, res) => {
    try {
      if (!req.body.vendorId || !req.body.imageIds) {
        return genResponse(res, HTTP_STATUS_CODES.ERROR, RESPONSE_MSG.RPM);
      }
      const result = await vendorMedia.update(
        { active: false },
        {
          where: {
            vendor_id: req.body.vendorId,
            id: req.body.imageIds, // Only delete images with IDs in the provided array
          },
        }
      );

      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not deleted");
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS_DEL);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  };

  deleteVendorAlbum = async (req, res) => {
    try {
      if (!req.body.vendorId || !req.body.albumIds) {
        return genResponse(res, HTTP_STATUS_CODES.ERROR, RESPONSE_MSG.RPM);
      }
      const result = await albums.update(
        { active: false },
        {
          where: {
            vendor_id: req.body.vendorId,
            id: req.body.albumIds, // Only delete images with IDs in the provided array
          },
        }
      );

      if (isEmpty(result)) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, RESPONSE_MSG.NOT_FOUND, "not deleted");
      }
      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS_DEL);
    } catch (e) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, RESPONSE_MSG.SMW, e);
    }
  };

  updateVendorDetail = async (req, res) => {
    try {
      const { vendorId, businessName, name, businessPhone, facebookPage, instagramPage, websiteLink, locationId, address } = req.body;

      if (!vendorId) {
        return genResponse(res, HTTP_STATUS_CODES.BAD_REQUEST, RESPONSE_MSG.RPM, "");
      }

      const vendor = await vendors.findByPk(vendorId);
      if (!vendor) {
        return genResponse(res, HTTP_STATUS_CODES.NOT_FOUND, "Vendor not found", "");
      }

      const updatedVendorData = {
        business_name: businessName || vendor.business_name,
        name: name || vendor.name,
        business_phone: businessPhone || vendor.business_phone,
        facebook_page_link: facebookPage,
        instagram_page_link: instagramPage,
        website_link: websiteLink,
      };

      if (address) {
        // If address or locationId is provided, update the address data
        const addressData = {
          full_address: address || vendor.address.full_address,
          lat: "", // Update latitude if available
          log: "", // Update longitude if available
        };

        if (locationId !== undefined) {
          addressData.location_id = locationId;
        }
        // Update the address record associated with the vendor
        await addresses.update(addressData, { where: { id: vendor.address_id } });
      }

      // Update the vendor record with the new data
      const result = await vendor.update(updatedVendorData);

      return genResponse(res, HTTP_STATUS_CODES.OK, RESPONSE_MSG.SUCCESS, result);
    } catch (error) {
      return genResponse(res, HTTP_STATUS_CODES.INTERNAL_SERVER, error.message, "");
    }
  };
}

module.exports = VendorService;
